/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.util;

import com.google.common.primitives.Ints;
import com.google.protobuf.Message;
import java.io.ByteArrayInputStream;
import java.io.DataInputStream;
import java.io.EOFException;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.InterruptedIOException;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.net.InetSocketAddress;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.regex.Pattern;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.HadoopIllegalArgumentException;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.BlockLocation;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.PathFilter;
import org.apache.hadoop.fs.permission.FsAction;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.hbase.ClusterId;
import org.apache.hadoop.hbase.HColumnDescriptor;
import org.apache.hadoop.hbase.HConstants;
import org.apache.hadoop.hbase.HDFSBlocksDistribution;
import org.apache.hadoop.hbase.HRegionInfo;
import org.apache.hadoop.hbase.RemoteExceptionHandler;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.classification.InterfaceAudience;
import org.apache.hadoop.hbase.exceptions.DeserializationException;
import org.apache.hadoop.hbase.fs.HFileSystem;
import org.apache.hadoop.hbase.master.HMaster;
import org.apache.hadoop.hbase.protobuf.ProtobufUtil;
import org.apache.hadoop.hbase.protobuf.generated.FSProtos;
import org.apache.hadoop.hbase.regionserver.HRegion;
import org.apache.hadoop.hbase.regionserver.StoreFileInfo;
import org.apache.hadoop.hbase.security.AccessDeniedException;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.CancelableProgressable;
import org.apache.hadoop.hbase.util.EnvironmentEdgeManager;
import org.apache.hadoop.hbase.util.FSHDFSUtils;
import org.apache.hadoop.hbase.util.FSRegionScanner;
import org.apache.hadoop.hbase.util.FileSystemVersionException;
import org.apache.hadoop.hbase.util.HBaseFsck;
import org.apache.hadoop.hdfs.DFSClient;
import org.apache.hadoop.hdfs.DFSHedgedReadMetrics;
import org.apache.hadoop.hdfs.DistributedFileSystem;
import org.apache.hadoop.hdfs.protocol.HdfsConstants;
import org.apache.hadoop.io.IOUtils;
import org.apache.hadoop.io.SequenceFile;
import org.apache.hadoop.ipc.RemoteException;
import org.apache.hadoop.security.UserGroupInformation;
import org.apache.hadoop.util.Progressable;
import org.apache.hadoop.util.ReflectionUtils;
import org.apache.hadoop.util.StringUtils;

@InterfaceAudience.Private
public abstract class FSUtils {
    private static final Log LOG = LogFactory.getLog(FSUtils.class);
    public static final String FULL_RWX_PERMISSIONS = "777";
    private static final String THREAD_POOLSIZE = "hbase.client.localityCheck.threadPoolSize";
    private static final int DEFAULT_THREAD_POOLSIZE = 2;
    public static final boolean WINDOWS = System.getProperty("os.name").startsWith("Windows");

    protected FSUtils() {
    }

    public static void setStoragePolicy(FileSystem fs, Configuration conf, Path path, String policyKey, String defaultPolicy) {
        String storagePolicy = conf.get(policyKey, defaultPolicy).toUpperCase(Locale.ROOT);
        if (storagePolicy.equals(defaultPolicy)) {
            if (LOG.isTraceEnabled()) {
                LOG.trace((Object)("default policy of " + defaultPolicy + " requested, exiting early."));
            }
            return;
        }
        if (fs instanceof DistributedFileSystem) {
            DistributedFileSystem dfs = (DistributedFileSystem)fs;
            Class<?> dfsClass = dfs.getClass();
            Method m = null;
            try {
                m = dfsClass.getDeclaredMethod("setStoragePolicy", Path.class, String.class);
                m.setAccessible(true);
            }
            catch (NoSuchMethodException e) {
                LOG.info((Object)"FileSystem doesn't support setStoragePolicy; --HDFS-6584 not available");
            }
            catch (SecurityException e) {
                LOG.info((Object)"Doesn't have access to setStoragePolicy on FileSystems --HDFS-6584 not available", (Throwable)e);
                m = null;
            }
            if (m != null) {
                try {
                    m.invoke((Object)dfs, path, storagePolicy);
                    LOG.info((Object)("set " + storagePolicy + " for " + path));
                }
                catch (Exception e) {
                    Throwable exception;
                    boolean probablyBadPolicy = false;
                    if (e instanceof InvocationTargetException && (exception = e.getCause()) instanceof RemoteException && HadoopIllegalArgumentException.class.getName().equals(((RemoteException)exception).getClassName())) {
                        LOG.warn((Object)("Given storage policy, '" + storagePolicy + "', was rejected and probably " + "isn't a valid policy for the version of Hadoop you're running. I.e. if you're " + "trying to use SSD related policies then you're likely missing HDFS-7228. For " + "more information see the 'ArchivalStorage' docs for your Hadoop release."));
                        LOG.debug((Object)"More information about the invalid storage policy.", exception);
                        probablyBadPolicy = true;
                    }
                    if (!probablyBadPolicy) {
                        LOG.warn((Object)("Unable to set " + storagePolicy + " for " + path), (Throwable)e);
                    }
                }
            }
        } else {
            LOG.info((Object)"FileSystem isn't an instance of DistributedFileSystem; presuming it doesn't support setStoragePolicy.");
        }
    }

    public static boolean isStartingWithPath(Path rootPath, String path) {
        String uriRootPath = rootPath.toUri().getPath();
        String tailUriPath = new Path(path).toUri().getPath();
        return tailUriPath.startsWith(uriRootPath);
    }

    public static boolean isMatchingTail(Path pathToSearch, String pathTail) {
        return FSUtils.isMatchingTail(pathToSearch, new Path(pathTail));
    }

    public static boolean isMatchingTail(Path pathToSearch, Path pathTail) {
        String toSearchName;
        String tailName;
        if (pathToSearch.depth() != pathTail.depth()) {
            return false;
        }
        Path tailPath = pathTail;
        Path toSearch = pathToSearch;
        boolean result = false;
        do {
            if ((tailName = tailPath.getName()) == null || tailName.length() <= 0) {
                result = true;
                break;
            }
            toSearchName = toSearch.getName();
            if (toSearchName == null || toSearchName.length() <= 0) break;
            tailPath = tailPath.getParent();
            toSearch = toSearch.getParent();
        } while (tailName.equals(toSearchName));
        return result;
    }

    public static FSUtils getInstance(FileSystem fs, Configuration conf) {
        String scheme = fs.getUri().getScheme();
        if (scheme == null) {
            LOG.warn((Object)("Could not find scheme for uri " + fs.getUri() + ", default to hdfs"));
            scheme = "hdfs";
        }
        Class fsUtilsClass = conf.getClass("hbase.fsutil." + scheme + ".impl", FSHDFSUtils.class);
        FSUtils fsUtils = (FSUtils)ReflectionUtils.newInstance((Class)fsUtilsClass, (Configuration)conf);
        return fsUtils;
    }

    public static boolean deleteDirectory(FileSystem fs, Path dir) throws IOException {
        return fs.exists(dir) && fs.delete(dir, true);
    }

    public static boolean deleteRegionDir(Configuration conf, HRegionInfo hri) throws IOException {
        Path rootDir = FSUtils.getRootDir(conf);
        FileSystem fs = rootDir.getFileSystem(conf);
        return FSUtils.deleteDirectory(fs, new Path(FSUtils.getTableDir(rootDir, hri.getTable()), hri.getEncodedName()));
    }

    public static long getDefaultBlockSize(FileSystem fs, Path path) throws IOException {
        Method m = null;
        Class<?> cls = fs.getClass();
        try {
            m = cls.getMethod("getDefaultBlockSize", Path.class);
        }
        catch (NoSuchMethodException e) {
            LOG.info((Object)"FileSystem doesn't support getDefaultBlockSize");
        }
        catch (SecurityException e) {
            LOG.info((Object)"Doesn't have access to getDefaultBlockSize on FileSystems", (Throwable)e);
            m = null;
        }
        if (m == null) {
            return fs.getDefaultBlockSize(path);
        }
        try {
            Object ret = m.invoke((Object)fs, path);
            return (Long)ret;
        }
        catch (Exception e) {
            throw new IOException(e);
        }
    }

    public static short getDefaultReplication(FileSystem fs, Path path) throws IOException {
        Method m = null;
        Class<?> cls = fs.getClass();
        try {
            m = cls.getMethod("getDefaultReplication", Path.class);
        }
        catch (NoSuchMethodException e) {
            LOG.info((Object)"FileSystem doesn't support getDefaultReplication");
        }
        catch (SecurityException e) {
            LOG.info((Object)"Doesn't have access to getDefaultReplication on FileSystems", (Throwable)e);
            m = null;
        }
        if (m == null) {
            return fs.getDefaultReplication(path);
        }
        try {
            Object ret = m.invoke((Object)fs, path);
            return ((Number)ret).shortValue();
        }
        catch (Exception e) {
            throw new IOException(e);
        }
    }

    public static int getDefaultBufferSize(FileSystem fs) {
        return fs.getConf().getInt("io.file.buffer.size", 4096);
    }

    public static FSDataOutputStream create(Configuration conf, FileSystem fs, Path path, FsPermission perm, InetSocketAddress[] favoredNodes) throws IOException {
        FileSystem backingFs;
        if (fs instanceof HFileSystem && (backingFs = ((HFileSystem)fs).getBackingFs()) instanceof DistributedFileSystem) {
            short replication = Short.parseShort(conf.get("DFS_REPLICATION", String.valueOf(0)));
            try {
                return (FSDataOutputStream)DistributedFileSystem.class.getDeclaredMethod("create", Path.class, FsPermission.class, Boolean.TYPE, Integer.TYPE, Short.TYPE, Long.TYPE, Progressable.class, InetSocketAddress[].class).invoke((Object)backingFs, path, perm, true, FSUtils.getDefaultBufferSize(backingFs), replication > 0 ? replication : FSUtils.getDefaultReplication(backingFs, path), FSUtils.getDefaultBlockSize(backingFs, path), null, favoredNodes);
            }
            catch (InvocationTargetException ite) {
                throw new IOException(ite.getCause());
            }
            catch (NoSuchMethodException e) {
                LOG.debug((Object)"DFS Client does not support most favored nodes create; using default create");
                if (LOG.isTraceEnabled()) {
                    LOG.trace((Object)"Ignoring; use default create", (Throwable)e);
                }
            }
            catch (IllegalArgumentException e) {
                LOG.debug((Object)("Ignoring (most likely Reflection related exception) " + e));
            }
            catch (SecurityException e) {
                LOG.debug((Object)("Ignoring (most likely Reflection related exception) " + e));
            }
            catch (IllegalAccessException e) {
                LOG.debug((Object)("Ignoring (most likely Reflection related exception) " + e));
            }
        }
        return FSUtils.create(fs, path, perm, true);
    }

    public static FSDataOutputStream create(FileSystem fs, Path path, FsPermission perm, boolean overwrite) throws IOException {
        if (LOG.isTraceEnabled()) {
            LOG.trace((Object)("Creating file=" + path + " with permission=" + perm + ", overwrite=" + overwrite));
        }
        return fs.create(path, perm, overwrite, FSUtils.getDefaultBufferSize(fs), FSUtils.getDefaultReplication(fs, path), FSUtils.getDefaultBlockSize(fs, path), null);
    }

    public static FsPermission getFilePermissions(FileSystem fs, Configuration conf, String permssionConfKey) {
        boolean enablePermissions = conf.getBoolean("hbase.data.umask.enable", false);
        if (enablePermissions) {
            try {
                FsPermission perm = new FsPermission(FULL_RWX_PERMISSIONS);
                String mask = conf.get(permssionConfKey);
                if (mask == null) {
                    return FsPermission.getFileDefault();
                }
                FsPermission umask = new FsPermission(mask);
                return perm.applyUMask(umask);
            }
            catch (IllegalArgumentException e) {
                LOG.warn((Object)("Incorrect umask attempted to be created: " + conf.get(permssionConfKey) + ", using default file permissions."), (Throwable)e);
                return FsPermission.getFileDefault();
            }
        }
        return FsPermission.getFileDefault();
    }

    public static void checkFileSystemAvailable(FileSystem fs) throws IOException {
        if (!(fs instanceof DistributedFileSystem)) {
            return;
        }
        IOException exception = null;
        DistributedFileSystem dfs = (DistributedFileSystem)fs;
        try {
            if (dfs.exists(new Path("/"))) {
                return;
            }
        }
        catch (IOException e) {
            exception = RemoteExceptionHandler.checkIOException((IOException)e);
        }
        try {
            fs.close();
        }
        catch (Exception e) {
            LOG.error((Object)"file system close failed: ", (Throwable)e);
        }
        IOException io = new IOException("File system is not available");
        io.initCause(exception);
        throw io;
    }

    private static boolean isInSafeMode(DistributedFileSystem dfs) throws IOException {
        boolean inSafeMode = false;
        try {
            Method m = DistributedFileSystem.class.getMethod("setSafeMode", HdfsConstants.SafeModeAction.class, Boolean.TYPE);
            inSafeMode = (Boolean)m.invoke((Object)dfs, HdfsConstants.SafeModeAction.SAFEMODE_GET, true);
        }
        catch (Exception e) {
            if (e instanceof IOException) {
                throw (IOException)e;
            }
            inSafeMode = dfs.setSafeMode(HdfsConstants.SafeModeAction.SAFEMODE_GET);
        }
        return inSafeMode;
    }

    public static void checkDfsSafeMode(Configuration conf) throws IOException {
        boolean isInSafeMode = false;
        FileSystem fs = FileSystem.get((Configuration)conf);
        if (fs instanceof DistributedFileSystem) {
            DistributedFileSystem dfs = (DistributedFileSystem)fs;
            isInSafeMode = FSUtils.isInSafeMode(dfs);
        }
        if (isInSafeMode) {
            throw new IOException("File system is in safemode, it can't be written now");
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static String getVersion(FileSystem fs, Path rootdir) throws IOException, DeserializationException {
        String version;
        block12: {
            Path versionFile = new Path(rootdir, "hbase.version");
            FileStatus[] status = null;
            try {
                status = fs.listStatus(versionFile);
            }
            catch (FileNotFoundException fnfe) {
                return null;
            }
            if (status == null || status.length == 0) {
                return null;
            }
            version = null;
            byte[] content = new byte[(int)status[0].getLen()];
            try (FSDataInputStream s = fs.open(versionFile);){
                IOUtils.readFully((InputStream)s, (byte[])content, (int)0, (int)content.length);
                if (ProtobufUtil.isPBMagicPrefix((byte[])content)) {
                    version = FSUtils.parseVersionFrom(content);
                    break block12;
                }
                ByteArrayInputStream is = new ByteArrayInputStream(content);
                try (DataInputStream dis = new DataInputStream(is);){
                    version = dis.readUTF();
                }
            }
        }
        return version;
    }

    static String parseVersionFrom(byte[] bytes) throws DeserializationException {
        ProtobufUtil.expectPBMagicPrefix((byte[])bytes);
        int pblen = ProtobufUtil.lengthOfPBMagic();
        FSProtos.HBaseVersionFileContent.Builder builder = FSProtos.HBaseVersionFileContent.newBuilder();
        try {
            ProtobufUtil.mergeFrom((Message.Builder)builder, (byte[])bytes, (int)pblen, (int)(bytes.length - pblen));
            return builder.getVersion();
        }
        catch (IOException e) {
            throw new DeserializationException((Throwable)e);
        }
    }

    static byte[] toVersionByteArray(String version) {
        FSProtos.HBaseVersionFileContent.Builder builder = FSProtos.HBaseVersionFileContent.newBuilder();
        return ProtobufUtil.prependPBMagic((byte[])builder.setVersion(version).build().toByteArray());
    }

    public static void checkVersion(FileSystem fs, Path rootdir, boolean message) throws IOException, DeserializationException {
        FSUtils.checkVersion(fs, rootdir, message, 0, 3);
    }

    public static void checkVersion(FileSystem fs, Path rootdir, boolean message, int wait, int retries) throws IOException, DeserializationException {
        String version = FSUtils.getVersion(fs, rootdir);
        if (version == null) {
            if (!FSUtils.metaRegionExists(fs, rootdir)) {
                FSUtils.setVersion(fs, rootdir, wait, retries);
                return;
            }
        } else if (version.compareTo("8") == 0) {
            return;
        }
        String msg = "HBase file layout needs to be upgraded. You have version " + version + " and I want version " + "8" + ". Consult http://hbase.apache.org/book.html for further information about upgrading HBase." + " Is your hbase.rootdir valid? If so, you may need to run " + "'hbase hbck -fixVersionFile'.";
        if (message) {
            System.out.println("WARNING! " + msg);
        }
        throw new FileSystemVersionException(msg);
    }

    public static void setVersion(FileSystem fs, Path rootdir) throws IOException {
        FSUtils.setVersion(fs, rootdir, "8", 0, 3);
    }

    public static void setVersion(FileSystem fs, Path rootdir, int wait, int retries) throws IOException {
        FSUtils.setVersion(fs, rootdir, "8", wait, retries);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void setVersion(FileSystem fs, Path rootdir, String version, int wait, int retries) throws IOException {
        Path versionFile = new Path(rootdir, "hbase.version");
        Path tempVersionFile = new Path(rootdir, ".tmp/hbase.version");
        while (true) {
            try {
                FSDataOutputStream s = fs.create(tempVersionFile);
                try {
                    s.write(FSUtils.toVersionByteArray(version));
                    s.close();
                    s = null;
                    if (!fs.rename(tempVersionFile, versionFile)) {
                        throw new IOException("Unable to move temp version file to " + versionFile);
                    }
                }
                finally {
                    try {
                        if (s != null) {
                            s.close();
                        }
                    }
                    catch (IOException ignore) {}
                }
                LOG.info((Object)("Created version file at " + rootdir.toString() + " with version=" + version));
                return;
            }
            catch (IOException e) {
                if (retries > 0) {
                    LOG.debug((Object)("Unable to create version file at " + rootdir.toString() + ", retrying"), (Throwable)e);
                    fs.delete(versionFile, false);
                    try {
                        if (wait > 0) {
                            Thread.sleep(wait);
                        }
                    }
                    catch (InterruptedException ie) {
                        throw (InterruptedIOException)new InterruptedIOException().initCause(ie);
                    }
                    --retries;
                    continue;
                }
                throw e;
            }
            break;
        }
    }

    public static boolean checkClusterIdExists(FileSystem fs, Path rootdir, int wait) throws IOException {
        while (true) {
            try {
                Path filePath = new Path(rootdir, "hbase.id");
                return fs.exists(filePath);
            }
            catch (IOException ioe) {
                if (wait > 0) {
                    LOG.warn((Object)("Unable to check cluster ID file in " + rootdir.toString() + ", retrying in " + wait + "msec: " + StringUtils.stringifyException((Throwable)ioe)));
                    try {
                        Thread.sleep(wait);
                    }
                    catch (InterruptedException e) {
                        throw (InterruptedIOException)new InterruptedIOException().initCause(e);
                    }
                    continue;
                }
                throw ioe;
            }
            break;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static ClusterId getClusterId(FileSystem fs, Path rootdir) throws IOException {
        FileStatus status;
        Path idPath = new Path(rootdir, "hbase.id");
        ClusterId clusterId = null;
        FileStatus fileStatus = status = fs.exists(idPath) ? fs.getFileStatus(idPath) : null;
        if (status != null) {
            int len = Ints.checkedCast((long)status.getLen());
            byte[] content = new byte[len];
            try (FSDataInputStream in = fs.open(idPath);){
                in.readFully(content);
            }
            try {
                clusterId = ClusterId.parseFrom((byte[])content);
            }
            catch (DeserializationException e) {
                throw new IOException("content=" + Bytes.toString((byte[])content), e);
            }
            if (!ProtobufUtil.isPBMagicPrefix((byte[])content)) {
                String cid = null;
                in = fs.open(idPath);
                try {
                    cid = in.readUTF();
                    clusterId = new ClusterId(cid);
                }
                catch (EOFException eof) {
                    LOG.warn((Object)("Cluster ID file " + idPath.toString() + " was empty"));
                }
                finally {
                    in.close();
                }
                FSUtils.rewriteAsPb(fs, rootdir, idPath, clusterId);
            }
            return clusterId;
        }
        LOG.warn((Object)("Cluster ID file does not exist at " + idPath.toString()));
        return clusterId;
    }

    private static void rewriteAsPb(FileSystem fs, Path rootdir, Path p, ClusterId cid) throws IOException {
        Path movedAsideName = new Path(p + "." + System.currentTimeMillis());
        if (!fs.rename(p, movedAsideName)) {
            throw new IOException("Failed rename of " + p);
        }
        FSUtils.setClusterId(fs, rootdir, cid, 100);
        if (!fs.delete(movedAsideName, false)) {
            throw new IOException("Failed delete of " + movedAsideName);
        }
        LOG.debug((Object)"Rewrote the hbase.id file as pb");
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void setClusterId(FileSystem fs, Path rootdir, ClusterId clusterId, int wait) throws IOException {
        while (true) {
            try {
                Path idFile = new Path(rootdir, "hbase.id");
                Path tempIdFile = new Path(rootdir, ".tmp/hbase.id");
                FSDataOutputStream s = fs.create(tempIdFile);
                try {
                    s.write(clusterId.toByteArray());
                    s.close();
                    s = null;
                    if (!fs.rename(tempIdFile, idFile)) {
                        throw new IOException("Unable to move temp version file to " + idFile);
                    }
                }
                finally {
                    try {
                        if (s != null) {
                            s.close();
                        }
                    }
                    catch (IOException ignore) {}
                }
                if (LOG.isDebugEnabled()) {
                    LOG.debug((Object)("Created cluster ID file at " + idFile.toString() + " with ID: " + clusterId));
                }
                return;
            }
            catch (IOException ioe) {
                if (wait > 0) {
                    LOG.warn((Object)("Unable to create cluster ID file in " + rootdir.toString() + ", retrying in " + wait + "msec: " + StringUtils.stringifyException((Throwable)ioe)));
                    try {
                        Thread.sleep(wait);
                    }
                    catch (InterruptedException e) {
                        throw (InterruptedIOException)new InterruptedIOException().initCause(e);
                    }
                    continue;
                }
                throw ioe;
            }
            break;
        }
    }

    public static Path validateRootPath(Path root) throws IOException {
        try {
            URI rootURI = new URI(root.toString());
            String scheme = rootURI.getScheme();
            if (scheme == null) {
                throw new IOException("Root directory does not have a scheme");
            }
            return root;
        }
        catch (URISyntaxException e) {
            IOException io = new IOException("Root directory path is not a valid URI -- check your hbase.rootdir configuration");
            io.initCause(e);
            throw io;
        }
    }

    public static String removeRootPath(Path path, Configuration conf) throws IOException {
        Path root = FSUtils.getRootDir(conf);
        String pathStr = path.toString();
        if (!pathStr.startsWith(root.toString())) {
            return pathStr;
        }
        return pathStr.substring(root.toString().length() + 1);
    }

    public static void waitOnSafeMode(Configuration conf, long wait) throws IOException {
        FileSystem fs = FileSystem.get((Configuration)conf);
        if (!(fs instanceof DistributedFileSystem)) {
            return;
        }
        DistributedFileSystem dfs = (DistributedFileSystem)fs;
        while (FSUtils.isInSafeMode(dfs)) {
            LOG.info((Object)"Waiting for dfs to exit safe mode...");
            try {
                Thread.sleep(wait);
            }
            catch (InterruptedException e) {
                throw (InterruptedIOException)new InterruptedIOException().initCause(e);
            }
        }
    }

    public static String getPath(Path p) {
        return p.toUri().getPath();
    }

    public static Path getRootDir(Configuration c) throws IOException {
        Path p = new Path(c.get("hbase.rootdir"));
        FileSystem fs = p.getFileSystem(c);
        return p.makeQualified(fs);
    }

    public static void setRootDir(Configuration c, Path root) throws IOException {
        c.set("hbase.rootdir", root.toString());
    }

    public static void setFsDefault(Configuration c, Path root) throws IOException {
        c.set("fs.defaultFS", root.toString());
        c.set("fs.default.name", root.toString());
    }

    public static boolean metaRegionExists(FileSystem fs, Path rootdir) throws IOException {
        Path metaRegionDir = HRegion.getRegionDir(rootdir, HRegionInfo.FIRST_META_REGIONINFO);
        return fs.exists(metaRegionDir);
    }

    public static HDFSBlocksDistribution computeHDFSBlocksDistribution(FileSystem fs, FileStatus status, long start, long length) throws IOException {
        BlockLocation[] blockLocations;
        HDFSBlocksDistribution blocksDistribution = new HDFSBlocksDistribution();
        for (BlockLocation bl : blockLocations = fs.getFileBlockLocations(status, start, length)) {
            String[] hosts = bl.getHosts();
            long len = bl.getLength();
            blocksDistribution.addHostsAndBlockWeight(hosts, len);
        }
        return blocksDistribution;
    }

    public static boolean isMajorCompacted(FileSystem fs, Path hbaseRootDir) throws IOException {
        List<Path> tableDirs = FSUtils.getTableDirs(fs, hbaseRootDir);
        RegionDirFilter regionFilter = new RegionDirFilter(fs);
        FamilyDirFilter familyFilter = new FamilyDirFilter(fs);
        for (Path d : tableDirs) {
            FileStatus[] regionDirs;
            for (FileStatus regionDir : regionDirs = fs.listStatus(d, (PathFilter)regionFilter)) {
                FileStatus[] familyDirs;
                Path dd = regionDir.getPath();
                for (FileStatus familyDir : familyDirs = fs.listStatus(dd, (PathFilter)familyFilter)) {
                    Path family = familyDir.getPath();
                    FileStatus[] familyStatus = fs.listStatus(family);
                    if (familyStatus.length <= 1) continue;
                    LOG.debug((Object)(family.toString() + " has " + familyStatus.length + " files."));
                    return false;
                }
            }
        }
        return true;
    }

    public static int getTotalTableFragmentation(HMaster master) throws IOException {
        Map<String, Integer> map = FSUtils.getTableFragmentation(master);
        return map != null && map.size() > 0 ? map.get("-TOTAL-") : -1;
    }

    public static Map<String, Integer> getTableFragmentation(HMaster master) throws IOException {
        Path path = FSUtils.getRootDir(master.getConfiguration());
        FileSystem fs = path.getFileSystem(master.getConfiguration());
        return FSUtils.getTableFragmentation(fs, path);
    }

    public static Map<String, Integer> getTableFragmentation(FileSystem fs, Path hbaseRootDir) throws IOException {
        HashMap<String, Integer> frags = new HashMap<String, Integer>();
        int cfCountTotal = 0;
        int cfFragTotal = 0;
        RegionDirFilter regionFilter = new RegionDirFilter(fs);
        FamilyDirFilter familyFilter = new FamilyDirFilter(fs);
        List<Path> tableDirs = FSUtils.getTableDirs(fs, hbaseRootDir);
        for (Path d : tableDirs) {
            FileStatus[] regionDirs;
            int cfCount = 0;
            int cfFrag = 0;
            for (FileStatus regionDir : regionDirs = fs.listStatus(d, (PathFilter)regionFilter)) {
                FileStatus[] familyDirs;
                Path dd = regionDir.getPath();
                for (FileStatus familyDir : familyDirs = fs.listStatus(dd, (PathFilter)familyFilter)) {
                    ++cfCount;
                    ++cfCountTotal;
                    Path family = familyDir.getPath();
                    FileStatus[] familyStatus = fs.listStatus(family);
                    if (familyStatus.length <= 1) continue;
                    ++cfFrag;
                    ++cfFragTotal;
                }
            }
            frags.put(FSUtils.getTableName(d).getNameAsString(), cfCount == 0 ? 0 : Math.round((float)cfFrag / (float)cfCount * 100.0f));
        }
        frags.put("-TOTAL-", cfCountTotal == 0 ? 0 : Math.round((float)cfFragTotal / (float)cfCountTotal * 100.0f));
        return frags;
    }

    public static Path getTableDir(Path rootdir, TableName tableName) {
        return new Path(FSUtils.getNamespaceDir(rootdir, tableName.getNamespaceAsString()), tableName.getQualifierAsString());
    }

    public static TableName getTableName(Path tablePath) {
        return TableName.valueOf((String)tablePath.getParent().getName(), (String)tablePath.getName());
    }

    public static Path getNamespaceDir(Path rootdir, String namespace) {
        return new Path(rootdir, new Path("data", new Path(namespace)));
    }

    public static boolean isAppendSupported(Configuration conf) {
        boolean append = conf.getBoolean("dfs.support.append", false);
        if (append) {
            try {
                SequenceFile.Writer.class.getMethod("syncFs", new Class[0]);
                append = true;
            }
            catch (SecurityException e) {
            }
            catch (NoSuchMethodException e) {
                append = false;
            }
        }
        if (!append) {
            try {
                FSDataOutputStream.class.getMethod("hflush", new Class[0]);
                append = true;
            }
            catch (NoSuchMethodException e) {
                append = false;
            }
        }
        return append;
    }

    public static boolean isHDFS(Configuration conf) throws IOException {
        FileSystem fs = FileSystem.get((Configuration)conf);
        String scheme = fs.getUri().getScheme();
        return scheme.equalsIgnoreCase("hdfs");
    }

    public abstract void recoverFileLease(FileSystem var1, Path var2, Configuration var3, CancelableProgressable var4) throws IOException;

    public static List<Path> getTableDirs(FileSystem fs, Path rootdir) throws IOException {
        LinkedList<Path> tableDirs = new LinkedList<Path>();
        for (FileStatus status : fs.globStatus(new Path(rootdir, new Path("data", "*")))) {
            tableDirs.addAll(FSUtils.getLocalTableDirs(fs, status.getPath()));
        }
        return tableDirs;
    }

    public static List<Path> getLocalTableDirs(FileSystem fs, Path rootdir) throws IOException {
        FileStatus[] dirs = fs.listStatus(rootdir, (PathFilter)new UserTableDirFilter(fs));
        ArrayList<Path> tabledirs = new ArrayList<Path>(dirs.length);
        for (FileStatus dir : dirs) {
            tabledirs.add(dir.getPath());
        }
        return tabledirs;
    }

    public static boolean isRecoveredEdits(Path path) {
        return path.toString().contains("recovered.edits");
    }

    public static List<Path> getRegionDirs(FileSystem fs, Path tableDir) throws IOException {
        FileStatus[] rds = fs.listStatus(tableDir, (PathFilter)new RegionDirFilter(fs));
        ArrayList<Path> regionDirs = new ArrayList<Path>(rds.length);
        for (FileStatus rdfs : rds) {
            Path rdPath = rdfs.getPath();
            regionDirs.add(rdPath);
        }
        return regionDirs;
    }

    public static List<Path> getFamilyDirs(FileSystem fs, Path regionDir) throws IOException {
        FileStatus[] fds = fs.listStatus(regionDir, (PathFilter)new FamilyDirFilter(fs));
        ArrayList<Path> familyDirs = new ArrayList<Path>(fds.length);
        for (FileStatus fdfs : fds) {
            Path fdPath = fdfs.getPath();
            familyDirs.add(fdPath);
        }
        return familyDirs;
    }

    public static List<Path> getReferenceFilePaths(FileSystem fs, Path familyDir) throws IOException {
        FileStatus[] fds = fs.listStatus(familyDir, (PathFilter)new ReferenceFileFilter(fs));
        ArrayList<Path> referenceFiles = new ArrayList<Path>(fds.length);
        for (FileStatus fdfs : fds) {
            Path fdPath = fdfs.getPath();
            referenceFiles.add(fdPath);
        }
        return referenceFiles;
    }

    public static FileSystem getCurrentFileSystem(Configuration conf) throws IOException {
        return FSUtils.getRootDir(conf).getFileSystem(conf);
    }

    public static Map<String, Path> getTableStoreFilePathMap(Map<String, Path> map, FileSystem fs, Path hbaseRootDir, TableName tableName) throws IOException {
        return FSUtils.getTableStoreFilePathMap(map, fs, hbaseRootDir, tableName, null);
    }

    public static Map<String, Path> getTableStoreFilePathMap(Map<String, Path> map, FileSystem fs, Path hbaseRootDir, TableName tableName, HBaseFsck.ErrorReporter errors) throws IOException {
        FileStatus[] regionDirs;
        if (map == null) {
            map = new HashMap<String, Path>();
        }
        Path tableDir = FSUtils.getTableDir(hbaseRootDir, tableName);
        FamilyDirFilter familyFilter = new FamilyDirFilter(fs);
        for (FileStatus regionDir : regionDirs = fs.listStatus(tableDir, (PathFilter)new RegionDirFilter(fs))) {
            FileStatus[] familyDirs;
            if (null != errors) {
                errors.progress();
            }
            Path dd = regionDir.getPath();
            for (FileStatus familyDir : familyDirs = fs.listStatus(dd, (PathFilter)familyFilter)) {
                FileStatus[] familyStatus;
                Path family;
                if (null != errors) {
                    errors.progress();
                }
                if ((family = familyDir.getPath()).getName().equals("recovered.edits")) continue;
                for (FileStatus sfStatus : familyStatus = fs.listStatus(family)) {
                    if (null != errors) {
                        errors.progress();
                    }
                    Path sf = sfStatus.getPath();
                    map.put(sf.getName(), sf);
                }
            }
        }
        return map;
    }

    public static int getRegionReferenceFileCount(FileSystem fs, Path p) {
        int result = 0;
        try {
            for (Path familyDir : FSUtils.getFamilyDirs(fs, p)) {
                result += FSUtils.getReferenceFilePaths(fs, familyDir).size();
            }
        }
        catch (IOException e) {
            LOG.warn((Object)"Error Counting reference files.", (Throwable)e);
        }
        return result;
    }

    public static Map<String, Path> getTableStoreFilePathMap(FileSystem fs, Path hbaseRootDir) throws IOException {
        return FSUtils.getTableStoreFilePathMap(fs, hbaseRootDir, null);
    }

    public static Map<String, Path> getTableStoreFilePathMap(FileSystem fs, Path hbaseRootDir, HBaseFsck.ErrorReporter errors) throws IOException {
        HashMap<String, Path> map = new HashMap<String, Path>();
        for (Path tableDir : FSUtils.getTableDirs(fs, hbaseRootDir)) {
            FSUtils.getTableStoreFilePathMap(map, fs, hbaseRootDir, FSUtils.getTableName(tableDir), errors);
        }
        return map;
    }

    public static FileStatus[] listStatus(FileSystem fs, Path dir, PathFilter filter) throws IOException {
        FileStatus[] status;
        block3: {
            status = null;
            try {
                status = filter == null ? fs.listStatus(dir) : fs.listStatus(dir, filter);
            }
            catch (FileNotFoundException fnfe) {
                if (!LOG.isTraceEnabled()) break block3;
                LOG.trace((Object)(dir + " doesn't exist"));
            }
        }
        if (status == null || status.length < 1) {
            return null;
        }
        return status;
    }

    public static FileStatus[] listStatus(FileSystem fs, Path dir) throws IOException {
        return FSUtils.listStatus(fs, dir, null);
    }

    public static boolean delete(FileSystem fs, Path path, boolean recursive) throws IOException {
        return fs.delete(path, recursive);
    }

    public static boolean isExists(FileSystem fs, Path path) throws IOException {
        return fs.exists(path);
    }

    public static void checkAccess(UserGroupInformation ugi, FileStatus file, FsAction action) throws AccessDeniedException {
        if (ugi.getShortUserName().equals(file.getOwner()) ? file.getPermission().getUserAction().implies(action) : (FSUtils.contains(ugi.getGroupNames(), file.getGroup()) ? file.getPermission().getGroupAction().implies(action) : file.getPermission().getOtherAction().implies(action))) {
            return;
        }
        throw new AccessDeniedException("Permission denied: action=" + action + " path=" + file.getPath() + " user=" + ugi.getShortUserName());
    }

    private static boolean contains(String[] groups, String user) {
        for (String group : groups) {
            if (!group.equals(user)) continue;
            return true;
        }
        return false;
    }

    public static void logFileSystemState(FileSystem fs, Path root, Log LOG) throws IOException {
        LOG.debug((Object)"Current file system:");
        FSUtils.logFSTree(LOG, fs, root, "|-");
    }

    private static void logFSTree(Log LOG, FileSystem fs, Path root, String prefix) throws IOException {
        FileStatus[] files = FSUtils.listStatus(fs, root, null);
        if (files == null) {
            return;
        }
        for (FileStatus file : files) {
            if (file.isDirectory()) {
                LOG.debug((Object)(prefix + file.getPath().getName() + "/"));
                FSUtils.logFSTree(LOG, fs, file.getPath(), prefix + "---");
                continue;
            }
            LOG.debug((Object)(prefix + file.getPath().getName()));
        }
    }

    public static boolean renameAndSetModifyTime(FileSystem fs, Path src, Path dest) throws IOException {
        fs.setTimes(src, EnvironmentEdgeManager.currentTime(), -1L);
        return fs.rename(src, dest);
    }

    public static Map<String, Map<String, Float>> getRegionDegreeLocalityMappingFromFS(Configuration conf) throws IOException {
        return FSUtils.getRegionDegreeLocalityMappingFromFS(conf, null, conf.getInt(THREAD_POOLSIZE, 2));
    }

    public static Map<String, Map<String, Float>> getRegionDegreeLocalityMappingFromFS(Configuration conf, String desiredTable, int threadPoolSize) throws IOException {
        ConcurrentHashMap<String, Map<String, Float>> regionDegreeLocalityMapping = new ConcurrentHashMap<String, Map<String, Float>>();
        FSUtils.getRegionLocalityMappingFromFS(conf, desiredTable, threadPoolSize, null, regionDegreeLocalityMapping);
        return regionDegreeLocalityMapping;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static void getRegionLocalityMappingFromFS(Configuration conf, String desiredTable, int threadPoolSize, Map<String, String> regionToBestLocalityRSMapping, Map<String, Map<String, Float>> regionDegreeLocalityMapping) throws IOException {
        PathFilter pathFilter;
        FileSystem fs = FileSystem.get((Configuration)conf);
        Path rootPath = FSUtils.getRootDir(conf);
        long startTime = EnvironmentEdgeManager.currentTime();
        Path queryPath = null == desiredTable ? new Path(new Path(rootPath, "data").toString() + "/*/*/*/") : new Path(FSUtils.getTableDir(rootPath, TableName.valueOf((String)desiredTable)).toString() + "/*/");
        FileStatus[] statusList = fs.globStatus(queryPath, pathFilter = new PathFilter(){

            public boolean accept(Path path) {
                if (null == path) {
                    return false;
                }
                Path parent = path.getParent();
                if (null == parent) {
                    return false;
                }
                String regionName = path.getName();
                if (null == regionName) {
                    return false;
                }
                return regionName.toLowerCase(Locale.ROOT).matches("[0-9a-f]+");
            }
        });
        if (null == statusList) {
            return;
        }
        LOG.debug((Object)("Query Path: " + queryPath + " ; # list of files: " + statusList.length));
        threadPoolSize = Math.min(threadPoolSize, statusList.length);
        ThreadPoolExecutor tpe = new ThreadPoolExecutor(threadPoolSize, threadPoolSize, 60L, TimeUnit.SECONDS, new ArrayBlockingQueue<Runnable>(statusList.length));
        try {
            for (FileStatus regionStatus : statusList) {
                Path regionPath;
                if (null == regionStatus || !regionStatus.isDirectory() || null == (regionPath = regionStatus.getPath())) continue;
                tpe.execute(new FSRegionScanner(fs, regionPath, regionToBestLocalityRSMapping, regionDegreeLocalityMapping));
            }
        }
        finally {
            tpe.shutdown();
            int threadWakeFrequency = conf.getInt("hbase.server.thread.wakefrequency", 60000);
            try {
                while (!tpe.awaitTermination(threadWakeFrequency, TimeUnit.MILLISECONDS)) {
                    LOG.info((Object)("Locality checking is underway: { Scanned Regions : " + tpe.getCompletedTaskCount() + "/" + tpe.getTaskCount() + " }"));
                }
            }
            catch (InterruptedException e) {
                throw (InterruptedIOException)new InterruptedIOException().initCause(e);
            }
        }
        long overhead = EnvironmentEdgeManager.currentTime() - startTime;
        String overheadMsg = "Scan DFS for locality info takes " + overhead + " ms";
        LOG.info((Object)overheadMsg);
    }

    public static void setupShortCircuitRead(Configuration conf) {
        boolean shortCircuitSkipChecksum = conf.getBoolean("dfs.client.read.shortcircuit.skip.checksum", false);
        boolean useHBaseChecksum = conf.getBoolean("hbase.regionserver.checksum.verify", true);
        if (shortCircuitSkipChecksum) {
            LOG.warn((Object)("Configuration \"dfs.client.read.shortcircuit.skip.checksum\" should not be set to true." + (useHBaseChecksum ? " HBase checksum doesn't require it, see https://issues.apache.org/jira/browse/HBASE-6868." : "")));
            assert (!shortCircuitSkipChecksum);
        }
        FSUtils.checkShortCircuitReadBufferSize(conf);
    }

    public static void checkShortCircuitReadBufferSize(Configuration conf) {
        int defaultSize = 131072;
        int notSet = -1;
        String dfsKey = "dfs.client.read.shortcircuit.buffer.size";
        int size = conf.getInt("dfs.client.read.shortcircuit.buffer.size", -1);
        if (size != -1) {
            return;
        }
        int hbaseSize = conf.getInt("hbase.dfs.client.read.shortcircuit.buffer.size", 131072);
        conf.setIfUnset("dfs.client.read.shortcircuit.buffer.size", Integer.toString(hbaseSize));
    }

    public static DFSHedgedReadMetrics getDFSHedgedReadMetrics(Configuration c) throws IOException {
        Method m;
        if (!FSUtils.isHDFS(c)) {
            return null;
        }
        String name = "getHedgedReadMetrics";
        DFSClient dfsclient = ((DistributedFileSystem)FileSystem.get((Configuration)c)).getClient();
        try {
            m = dfsclient.getClass().getDeclaredMethod("getHedgedReadMetrics", new Class[0]);
        }
        catch (NoSuchMethodException e) {
            LOG.warn((Object)("Failed find method getHedgedReadMetrics in dfsclient; no hedged read metrics: " + e.getMessage()));
            return null;
        }
        catch (SecurityException e) {
            LOG.warn((Object)("Failed find method getHedgedReadMetrics in dfsclient; no hedged read metrics: " + e.getMessage()));
            return null;
        }
        m.setAccessible(true);
        try {
            return (DFSHedgedReadMetrics)m.invoke((Object)dfsclient, new Object[0]);
        }
        catch (IllegalAccessException e) {
            LOG.warn((Object)("Failed invoking method getHedgedReadMetrics on dfsclient; no hedged read metrics: " + e.getMessage()));
            return null;
        }
        catch (IllegalArgumentException e) {
            LOG.warn((Object)("Failed invoking method getHedgedReadMetrics on dfsclient; no hedged read metrics: " + e.getMessage()));
            return null;
        }
        catch (InvocationTargetException e) {
            LOG.warn((Object)("Failed invoking method getHedgedReadMetrics on dfsclient; no hedged read metrics: " + e.getMessage()));
            return null;
        }
    }

    public static class ReferenceFileFilter
    implements PathFilter {
        private final FileSystem fs;

        public ReferenceFileFilter(FileSystem fs) {
            this.fs = fs;
        }

        public boolean accept(Path rd) {
            try {
                return !this.fs.getFileStatus(rd).isDirectory() && StoreFileInfo.isReference(rd);
            }
            catch (IOException ioe) {
                LOG.warn((Object)("Skipping file " + rd + " due to IOException"), (Throwable)ioe);
                return false;
            }
        }
    }

    public static class HFileFilter
    implements PathFilter {
        final FileSystem fs;

        public HFileFilter(FileSystem fs) {
            this.fs = fs;
        }

        public boolean accept(Path rd) {
            try {
                return !this.fs.getFileStatus(rd).isDirectory() && StoreFileInfo.isHFile(rd);
            }
            catch (IOException ioe) {
                LOG.warn((Object)("Skipping file " + rd + " due to IOException"), (Throwable)ioe);
                return false;
            }
        }
    }

    public static class FamilyDirFilter
    implements PathFilter {
        final FileSystem fs;

        public FamilyDirFilter(FileSystem fs) {
            this.fs = fs;
        }

        public boolean accept(Path rd) {
            try {
                HColumnDescriptor.isLegalFamilyName((byte[])Bytes.toBytes((String)rd.getName()));
            }
            catch (IllegalArgumentException iae) {
                return false;
            }
            try {
                return this.fs.getFileStatus(rd).isDirectory();
            }
            catch (IOException ioe) {
                LOG.warn((Object)("Skipping file " + rd + " due to IOException"), (Throwable)ioe);
                return false;
            }
        }
    }

    public static class RegionDirFilter
    implements PathFilter {
        public static final Pattern regionDirPattern = Pattern.compile("^[0-9a-f]*$");
        final FileSystem fs;

        public RegionDirFilter(FileSystem fs) {
            this.fs = fs;
        }

        public boolean accept(Path rd) {
            if (!regionDirPattern.matcher(rd.getName()).matches()) {
                return false;
            }
            try {
                return this.fs.getFileStatus(rd).isDirectory();
            }
            catch (IOException ioe) {
                LOG.warn((Object)("Skipping file " + rd + " due to IOException"), (Throwable)ioe);
                return false;
            }
        }
    }

    public static class UserTableDirFilter
    extends BlackListDirFilter {
        public UserTableDirFilter(FileSystem fs) {
            super(fs, HConstants.HBASE_NON_TABLE_DIRS);
        }

        @Override
        protected boolean isValidName(String name) {
            if (!super.isValidName(name)) {
                return false;
            }
            try {
                TableName.isLegalTableQualifierName((byte[])Bytes.toBytes((String)name));
            }
            catch (IllegalArgumentException e) {
                LOG.info((Object)("INVALID NAME " + name));
                return false;
            }
            return true;
        }
    }

    public static class DirFilter
    extends BlackListDirFilter {
        public DirFilter(FileSystem fs) {
            super(fs, null);
        }
    }

    public static class BlackListDirFilter
    implements PathFilter {
        private final FileSystem fs;
        private List<String> blacklist;

        public BlackListDirFilter(FileSystem fs, List<String> directoryNameBlackList) {
            this.fs = fs;
            this.blacklist = directoryNameBlackList == null ? Collections.emptyList() : directoryNameBlackList;
        }

        public boolean accept(Path p) {
            boolean isValid = false;
            try {
                isValid = this.isValidName(p.getName()) ? this.fs.getFileStatus(p).isDirectory() : false;
            }
            catch (IOException e) {
                LOG.warn((Object)("An error occurred while verifying if [" + p.toString() + "] is a valid directory. Returning 'not valid' and continuing."), (Throwable)e);
            }
            return isValid;
        }

        protected boolean isValidName(String name) {
            return !this.blacklist.contains(name);
        }
    }

    static class FileFilter
    implements PathFilter {
        private final FileSystem fs;

        public FileFilter(FileSystem fs) {
            this.fs = fs;
        }

        public boolean accept(Path p) {
            try {
                return this.fs.isFile(p);
            }
            catch (IOException e) {
                LOG.debug((Object)("unable to verify if path=" + p + " is a regular file"), (Throwable)e);
                return false;
            }
        }
    }
}

